import os
import re
import time
from flask import Flask, request, abort, render_template, jsonify, Response
import telebot

from config import load_config
from db import DB
from security import validate_webapp_init_data

def now() -> int:
    return int(time.time())

def create_app() -> Flask:
    cfg = load_config()
    os.makedirs(cfg.data_dir, exist_ok=True)
    db = DB(os.path.join(cfg.data_dir, "db.sqlite3"))

    bot = telebot.TeleBot(cfg.bot_token, parse_mode="HTML", threaded=False)
    from bot_logic import BotApp
    BotApp(bot=bot, db=db, base_url=cfg.base_url, data_dir=cfg.data_dir)

    app = Flask(__name__)
    app.config["MAX_CONTENT_LENGTH"] = 1024 * 1024 * 1024  # 1GB (adjust if needed)

    @app.get("/health")
    def health():
        return "ok"

    @app.post(f"/webhook/{cfg.webhook_secret}")
    def webhook():
        if request.headers.get("content-type") != "application/json":
            abort(415)
        update = telebot.types.Update.de_json(request.data.decode("utf-8"))
        bot.process_new_updates([update])
        return "OK"

    @app.get("/wa")
    def wa():
        token = (request.args.get("token") or "").strip()
        return render_template("wa.html", token=token)

    @app.post("/api/media")
    def api_media():
        body = request.get_json(silent=True) or {}
        token = (body.get("token") or "").strip()
        init_data = (body.get("initData") or "").strip()

        if not token:
            return jsonify({"ok": False, "error": "NO_TOKEN"}), 403

        trow = db.get_token(token)
        if not trow or int(trow["expires_at"]) < now():
            return jsonify({"ok": False, "error": "TOKEN_EXPIRED"}), 403

        parsed = validate_webapp_init_data(init_data, cfg.bot_token)
        if not parsed or not parsed.get("user") or not parsed["user"].get("id"):
            return jsonify({"ok": False, "error": "BAD_INITDATA"}), 403

        tg_id = int(parsed["user"]["id"])
        if tg_id != int(trow["tg_id"]):
            return jsonify({"ok": False, "error": "NOT_ALLOWED"}), 403

        media = db.get_media(int(trow["media_id"]))
        if not media:
            return jsonify({"ok": False, "error": "MEDIA_NOT_FOUND"}), 404

        webtxt = db.get_setting("webapp_text", " ")
        return jsonify({
            "ok": True,
            "title": f"کیفیت: {media['quality']}",
            "text": webtxt,
            "src": f"/stream/{token}"
        })

    def send_file_partial(path: str):
        # Basic Range support for mp4 seeking
        file_size = os.path.getsize(path)
        range_header = request.headers.get("Range", None)
        if not range_header:
            def generate():
                with open(path, "rb") as f:
                    while True:
                        chunk = f.read(1024 * 1024)
                        if not chunk:
                            break
                        yield chunk
            return Response(generate(), mimetype="video/mp4",
                            headers={"Content-Length": str(file_size), "Accept-Ranges": "bytes"})

        m = re.match(r"bytes=(\d+)-(\d*)", range_header)
        if not m:
            abort(416)
        start = int(m.group(1))
        end = int(m.group(2)) if m.group(2) else file_size - 1
        end = min(end, file_size - 1)
        if start > end:
            abort(416)

        length = end - start + 1

        def generate():
            with open(path, "rb") as f:
                f.seek(start)
                remaining = length
                while remaining > 0:
                    chunk = f.read(min(1024 * 1024, remaining))
                    if not chunk:
                        break
                    remaining -= len(chunk)
                    yield chunk

        rv = Response(generate(), 206, mimetype="video/mp4")
        rv.headers.add("Content-Range", f"bytes {start}-{end}/{file_size}")
        rv.headers.add("Accept-Ranges", "bytes")
        rv.headers.add("Content-Length", str(length))
        rv.headers.add("Content-Disposition", "inline")
        rv.headers.add("Cache-Control", "no-store")
        return rv

    @app.get("/stream/<token>")
    def stream(token: str):
        token = (token or "").strip()
        trow = db.get_token(token)
        if not trow or int(trow["expires_at"]) < now():
            abort(403)
        media = db.get_media(int(trow["media_id"]))
        if not media:
            abort(404)
        path = media["path"]
        if not os.path.exists(path):
            abort(404)
        return send_file_partial(path)

    return app

app = create_app()